<?php
/**
 * ===================================================================================
 * Add Parent/Guardian Remark Page
 * ===================================================================================
 *
 * File: add_remark.php
 * Location: /homework_portal/student/add_remark.php
 *
 * Purpose:
 * 1. Provides a form for a parent/guardian to add remarks to a student's submission.
 * 2. Updates the 'parent_guardian_remark' field for the specific submission.
 */

require_once '../config.php';

// --- Security Check ---
if (!isset($_SESSION['student_id'])) {
    header('Location: ../login.php');
    exit();
}

$submission_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$errors = [];
$success_message = '';

if ($submission_id <= 0) {
    header('Location: dashboard.php');
    exit();
}

// --- Fetch Submission & Assignment Details ---
try {
    $stmt = $pdo->prepare("SELECT s.id, s.parent_guardian_remark, a.title, a.subject_id FROM submissions s JOIN assignments a ON s.assignment_id = a.id WHERE s.id = ? AND s.student_id = ?");
    $stmt->execute([$submission_id, $_SESSION['student_id']]);
    $submission = $stmt->fetch();
    if (!$submission) {
        header('Location: dashboard.php?error=Submission not found.');
        exit();
    }
} catch (PDOException $e) {
    die("Database error fetching submission data.");
}

// --- Handle Form Submission ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $remark = trim($_POST['remark'] ?? '');
    if (empty($remark)) {
        $errors[] = "Remark field cannot be empty.";
    }

    if (empty($errors)) {
        try {
            $sql = "UPDATE submissions SET parent_guardian_remark = ? WHERE id = ?";
            $pdo->prepare($sql)->execute([$remark, $submission_id]);
            $success_message = "Remark added successfully!";
            // Re-fetch data
            $stmt->execute([$submission_id, $_SESSION['student_id']]);
            $submission = $stmt->fetch();
        } catch (PDOException $e) {
            $errors[] = "Could not save your remark.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Parent/Guardian Remark</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600&family=Baloo+2:wght@700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f0f4f8; }
        .font-brand { font-family: 'Baloo 2', cursive; }
    </style>
</head>
<body class="min-h-screen">
     <main class="max-w-2xl mx-auto py-8 px-4">
        <div class="mb-4">
            <a href="view_subject.php?id=<?php echo e($submission['subject_id']); ?>" class="flex items-center text-cyan-700 hover:text-cyan-800 font-medium">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path></svg>
                Back to Subject
            </a>
        </div>
        <div class="bg-white p-8 rounded-lg shadow-lg">
            <h1 class="text-2xl font-bold text-gray-800 mb-2 font-brand">Parent/Guardian Remark</h1>
            <p class="text-gray-500 mb-6">For Assignment: <strong class="text-gray-700"><?php echo e($submission['title']); ?></strong></p>
            
            <?php if ($success_message): ?><div class="p-4 mb-4 text-sm text-green-700 bg-green-100 rounded-lg"><?php echo e($success_message); ?></div><?php endif; ?>
            <?php if (!empty($errors)): ?><div class="p-4 mb-4 text-sm text-red-700 bg-red-100 rounded-lg"><ul class="list-disc list-inside"><?php foreach ($errors as $e) echo "<li>" . e($e) . "</li>"; ?></ul></div><?php endif; ?>

            <form action="add_remark.php?id=<?php echo $submission_id; ?>" method="POST" class="space-y-4">
                <div>
                    <label for="remark" class="block text-sm font-medium text-gray-700">Your Remark</label>
                    <textarea name="remark" id="remark" rows="6" class="mt-1 block w-full p-2 border border-gray-300 rounded-md" placeholder="e.g., I have reviewed this work with my child."><?php echo e($submission['parent_guardian_remark']); ?></textarea>
                </div>
                <div class="flex justify-end pt-4 border-t">
                    <button type="submit" class="px-6 py-2 bg-teal-600 text-white font-semibold rounded-lg shadow hover:bg-teal-700">Save Remark</button>
                </div>
            </form>
        </div>
    </main>
</body>
</html>
